/**
 * MapArrayInt.mqh
 *
 * This content is released under the terms of the MIT license.
 * A copy of this license has been included with this project in LICENSE.txt
 */
#property copyright   "Copyright (c) 2015 TradertoolsFX"
#property link        "http://www.tradertools-fx.com"
#property strict

#include <Arrays\ArrayInt.mqh>

/**
 * Maps int keys to arrays of int values.
 *
 * Special-purpose, limited functionality.
 */
class MapArrayInt {
public:
    MapArrayInt(void);
    ~MapArrayInt(void);

    int GetKey(int index) const;
    int GetValue(int key, int index) const;

    int GetKeyIndex(int key) const;
    int GetValueCount(int key) const;
    int GetKeyCount(void) const;

    int AddValue(int key, int value);
private:
    CArrayInt m_keys;
    CArrayInt *m_values[];
    int m_key_count;

    int AddKey(int key);
};

MapArrayInt::MapArrayInt(void)
{
    ZeroMemory(m_values);
    m_key_count = 0;
}

MapArrayInt::~MapArrayInt(void)
{
    for (int i = 0; i < m_key_count; i++) {
        delete m_values[i];
    }
}

/* Return the key at index. If index is out of range, returns INT_MAX. */
int MapArrayInt::GetKey(int index) const
{
    return m_keys.At(index);
}

/* Return the value at the index of the values array mapped to key.
 * Returns INT_MAX if key does not exist or index out of range. */
int MapArrayInt::GetValue(int key, int index) const
{
    int key_index = GetKeyIndex(key);
    if (key_index == -1) {
        return INT_MAX;
    }

    return m_values[key_index].At(index);
}

/* Return the key index or -1 if it does not exist. */
int MapArrayInt::GetKeyIndex(int key) const
{
    for (int i = 0; i < m_key_count; i++) {
        if (m_keys.At(i) == key) {
            return i;
        }
    }

    return -1;
}

/* Return the count of the value array mapped to key.
 * Returns INT_MAX if the key does not exist. */
int MapArrayInt::GetValueCount(int key) const
{
    int index = GetKeyIndex(key);
    if (index == -1) {
        return INT_MAX;
    }

    return m_values[index].Total();
}

int MapArrayInt::GetKeyCount(void) const
{
    return m_key_count;
}

/* Add a value to the array mapped to key.
 * If the key does not exist, it adds a new key.
 * Returns the index of the newly added value or -1 on failure. */
int MapArrayInt::AddValue(int key, int value)
{
    int index = GetKeyIndex(key);
    if (index == -1) { // Key does not exist
        index = AddKey(key);
        if (index == -1) {
            return -1;
        }
    }

    if (!m_values[index].Add(value)) {
        return -1;
    }

    return m_values[index].Total() - 1; // Index of last item.
}

/* Add key and create new value array.
 * Returns the new key index or -1 on failure. */
int MapArrayInt::AddKey(int key)
{
    if (!m_keys.Add(key)) {
        return -1;
    }

    m_key_count++;
    int key_index = m_key_count - 1;
    int values_size = ArraySize(m_values);

    if (key_index >= values_size || values_size == 0) {
        if (values_size == 0) {
            values_size = 1;
        }

        if (ArrayResize(m_values, values_size * 2) == -1) {
            Print(__FUNCTION__,
                  ": Failed to resize array. Last error: ",
                  GetLastError());
            return -1;
        }
    }

    m_values[key_index] = new CArrayInt();

    return key_index;
}
