/* FC_FloatBySymbolTimeframe.mq4 - Float charts by symbol(s) and/or timeframe(s).
 * Float only EURUSD charts or
 * float both EURUSD and EURJPY charts.
 * Maybe you just want to float EURUSD,H4 charts.
 * Or maybe you just want to float H4 charts, regardless of symbol.
 * Any combination should work with this script.
 *
 * This content is released under the terms of the MIT license.
 * A copy of this license has been included with this project in LICENSE.txt
 */
#property copyright   "Copyright (c) 2015 TradertoolsFX"
#property link        "http://www.tradertools-fx.com"
#property version     "1.0"
#property strict
#property show_inputs

#include "inc/fcprecheck.mqh"

#define NUM_TIMEFRAMES 9

input string _desc_ = "===You can float all the charts of a given symbol(s) AND/OR period/timeframe(s)===";
input string _desc1_ = "===Enter BROKER symbol(s) separated by comma===";
input string _desc1_2_ = "=>If blank, all symbols will be floated per PeriodList input<=";
input string SymbolList;
input string _desc2_ = "===Enter period(s) as separated by comma===";
input string _desc2_2_ = "=>If blank, all timeframes will be floated per SymbolList input<=";
input string PeriodList;

string g_timeframes[][NUM_TIMEFRAMES] = {
    { "1", "M1", "1M", "1 minute", NULL }, // 1 minute
    { "5", "M5", "5M", "5 minute", NULL }, // 5 minute
    { "15", "M15", "15M", "15 minute", NULL }, // 15 minute
    { "30", "M30", "30M", "30 minute", NULL }, // 30 minute
    { "60", "H1", "1H", "Hourly", "Hour", NULL }, // 1 hour
    { "240", "H4", "4H", "4 Hour", NULL }, // 4 hour
    { "1440", "D1", "1D", "Daily", NULL }, // Daily
    { "10020", "W1", "1W", "Weekly", NULL }, // Weekly
    { "43200", "MN", "Monthly", NULL } // Monthly
};

void OnStart()
{
    FloatingCharts fc;
    if (FCPreCheck(fc) != FC_ERR_SUCCESS) {
        return;
    }

    string symbols[];
    int num_symbols = StringSplit(SymbolList, ',', symbols);
    TrimStrings(symbols, num_symbols);

    int periods[];
    int num_periods = StringsToPeriods(periods, PeriodList, ',', g_timeframes);
    if (num_periods == -1) {
        return;
    }

    FloatBySymbolPeriod(fc, symbols, num_symbols, periods, num_periods);
}

void FloatBySymbolPeriod(const FloatingCharts &fc,
                         const string &symbols[],
                         int num_symbols,
                         const int &periods[],
                         int num_periods)
{
    string symbol;
    int period;
    bool symbol_found, period_found;
    int i;

    long chart_id = ChartFirst();
    while (chart_id != -1 &&
           !IsStopped()) {
        if (!fc.IsChartFloating(chart_id)) {
            symbol = ChartSymbol(chart_id);
            symbol_found = (num_symbols == 0); // true if param left blank
            for (i = 0; i < num_symbols; i++) {
                if (StringCompare(symbol, symbols[i], false) == 0) {
                    symbol_found = true;
                    break;
                }
            }

            period = ChartPeriod(chart_id);
            period_found = (num_periods == 0); // true if param left blank
            for (i = 0; i < num_periods; i++) {
                if (period == periods[i]) {
                    period_found = true;
                    break;
                }
            }

            if (symbol_found && period_found) {
                fc.Float(chart_id);
            }
        }

        chart_id = ChartNext(chart_id);
    }
}

void TrimStrings(string &strings[], int num_strings)
{
    for (int i = 0; i < num_strings; i++) {
        strings[i] = StringTrimLeft(strings[i]);
        strings[i] = StringTrimRight(strings[i]);
    }
}

int PeriodStrToInt(string period_str, const string &timeframes[][NUM_TIMEFRAMES])
{
    int period = 0;
    int i, j;
    for (i = 0; i < NUM_TIMEFRAMES; i++) {
        for (j = 0; timeframes[i][j] != NULL; j++) {
            if (StringCompare(period_str, timeframes[i][j], false) == 0) {
                period = GetPeriodByIndex(i);
                break;
            }
        }

        if (period > 0) {
            break;
        }
    }

    return period;
}

int GetPeriodByIndex(int index)
{
    int period = 0;

    switch (index) {
    case 0:
        period = PERIOD_M1;
        break;
    case 1:
        period = PERIOD_M5;
        break;
    case 2:
        period = PERIOD_M15;
        break;
    case 3:
        period = PERIOD_M30;
        break;
    case 4:
        period = PERIOD_H1;
        break;
    case 5:
        period = PERIOD_H4;
        break;
    case 6:
        period = PERIOD_D1;
        break;
    case 7:
        period = PERIOD_W1;
        break;
    case 8:
        period = PERIOD_MN1;
        break;
    default:
        period = 0;
        break;
    }

    return period;
}

int StringsToPeriods(int &periods[],
                     const string str_list,
                     ushort delimiter,
                     const string &timeframes[][])
{
    string strings[];
    int n = StringSplit(str_list, delimiter, strings);
    if (ArrayResize(periods, n) == -1) {
        return -1;
    }
    TrimStrings(strings, n);

    int i;
    for (i = 0; i < n; i++) {
        periods[i] = PeriodStrToInt(strings[i], timeframes);
        if (periods[i] == 0) {
            Alert("Invalid input detected: ", strings[i],
                  " Period list must contain integers separated by a comma.");
            return -1;
        }
    }

    return i;
}

/*
int StringsToInts(int &ints[], const string &str, ushort delimiter)
{
    string strings[];
    int n = StringSplit(str, delimiter, strings);
    if (ArrayResize(ints, n) == -1) {
        return -1;
    }
    TrimStrings(strings, n);

    int i;
    for (i = 0; i < n; i++) {
        if (IsPositiveInteger(strings[i])) {
            ints[i] = (int)StringToInteger(strings[i]);
        } else {
            Alert("Invalid input detected: ", strings[i],
                  " Period list must contain integers separated by a comma.");
            return -1;
        }
    }

    return i;
}

bool IsPositiveInteger(const string &str)
{
    unsigned short char_code;
    for (int i = 0; i < StringLen(str); i++) {
        char_code = StringGetChar(str, i);
        if (char_code < '0' || char_code > '9') {
            return false;
        }
    }
    return true;
}
*/
