/**
 * @internal
 * @file version.mqh
 * Version struct definition and conversion functions.
 *
 * @license
 *  This content is released under the terms of the MIT license.
 *  A copy of this license has been included with this project in LICENSE.txt
 */
#property copyright "Copyright (c) 2015 TradertoolsFX"
#property link      "http://www.tradertools-fx.com"
#property strict

/**
 * @cond INTERNAL_STRUCT
 * @struct Version
 * 'major.minor.patch' version representation and construction.
 */
struct Version {
    int major;
    int minor;
    int patch;

    /**
     * Construct version.
     */
    Version(int v_major, int v_minor, int v_patch)
    {
        major = v_major;
        minor = v_minor;
        patch = v_patch;
    }

    /**
     * Construct version from calculated integer.
     *
     * @param int version_int - the calculated version integer.
     */
    Version(int version_int) {
        IntToVersion(version_int, this);
    }
};
/** @endcond */

/**
 * Convert a Version struct to a string of the form 'major.minor.patch'.
 *
 * @param Version version_struct
 *
 * @returns Version as a string.
 */
string VersionToString(const Version &version_struct)
{
    return StringFormat("%d.%d.%d",
                        version_struct.major,
                        version_struct.minor,
                        version_struct.patch);
}

/**
 * Convert a Version struct to a calculated integer.
 *
 * @param Version version_struct
 *
 * @return An integer calculated from the Version struct.
 */
int VersionToInt(const Version &version_struct)
{
    return (version_struct.major * 1000000) +
           (version_struct.minor * 1000) +
           version_struct.patch;
}

/**
 * Convert a version integer to a version struct.
 *
 * @param[in] int version_int
 * @param[out] Version version_struct
 */
void IntToVersion(int version_int, Version &version_struct)
{
    version_struct.major = version_int / 1000000;
    version_struct.minor = (version_int % 1000000) / 1000;
    version_struct.patch = version_int % 1000;
}

/**
 * Compare two Version structs.
 *
 * @param Version version1
 * @param Version version2
 *
 * @returns A negative value if version1 < version2.
 * @returns 0 if version1 = version2.
 * @returns A positive value if version1 > version2.
 */
int VersionCompare(const Version &version1, const Version &version2)
{
    int version1_int = VersionToInt(version1);
    int version2_int = VersionToInt(version2);

    return version1_int - version2_int;
}
