/**
 * @file fcerror.mqh
 * Error codes and error code conversion functions
 * used in the FloatingCharts API and its extensions.
 *
 * @license
 *  This content is released under the terms of the MIT license.
 *  A copy of this license has been included with this project in LICENSE.txt
 */
#property copyright "Copyright (c) 2015 TradertoolsFX"
#property link      "http://www.tradertools-fx.com"
#property strict

/**
 * Enumeration of Floating Charts API error codes.
 */
enum FCErrorCode {
    FC_ERR_SUCCESS, /**< No error. */
    FC_ERR_CHART_FLOATING, /**< Chart already floating.
                             *  Can only be applied to a docked chart. */
    FC_ERR_CHART_NOT_FLOATING, /**< Chart not floating (docked).
                                 *  Can only be applied to a floating chart. */
    FC_ERR_DLL_NOT_ALLOWED, /**< DLL imports are not allowed. */
    FC_ERR_NOT_RUNNING, /**< MT4 Floating Charts is not attached
                          *  to this instance of MetaTrader. */
    FC_ERR_INVALID_PARAMETER, /**< An invalid argument was encountered. */
    FC_ERR_INVALID_HANDLE, /**< A window handle is not set or
                             *  failed to obtain a window handle(s). */
    FC_ERR_UNSUPPORTED_VERSION, /**< The running version of
                                  *  MT4 Floating Charts  does not meet
                                  *  the minimum version required
                                  *  to continue. */
    FC_ERR_COMMAND_FAILED /**< A command sent to MT4 Floating Charts failed.
                            *  This can usually be interpreted as
                            *  FC_ERR_NOT_RUNNING because that is the
                            *  most common cause of command failure */
};

/**
 * Convert a Floating Charts API error code into a more descriptive string.
 *
 * @param int error_code
 *
 * @returns The named constant for the error code as a string.
 * @returns "Invalid error code" if error_code is not a #FCErrorCode.
 */
string FCErrorDescription(int error_code)
{
    string result;

    switch (error_code) {
    case FC_ERR_SUCCESS:
        result = "FC_ERR_SUCCESS";
        break;
    case FC_ERR_CHART_FLOATING:
        result = "FC_ERR_CHART_FLOATING";
        break;
    case FC_ERR_CHART_NOT_FLOATING:
        result = "FC_ERR_CHART_NOT_FLOATING";
        break;
    case FC_ERR_DLL_NOT_ALLOWED:
        result = "FC_ERR_DLL_NOT_ALLOWED";
        break;
    case FC_ERR_NOT_RUNNING:
        result = "FC_ERR_NOT_RUNNING";
        break;
    case FC_ERR_INVALID_PARAMETER:
        result = "FC_ERR_INVALID_PARAMETER";
        break;
    case FC_ERR_INVALID_HANDLE:
        result = "FC_ERR_INVALID_HANDLE";
        break;
    case FC_ERR_UNSUPPORTED_VERSION:
        result = "FC_ERR_UNSUPPORTED_VERSION";
        break;
    case FC_ERR_COMMAND_FAILED:
        result = "FC_ERR_COMMAND_FAILED";
        break;
    default:
        result = "Invalid error code";
        break;
    }

    return result;
}
