/**
 * @file FloatingCharts.mqh
 * The MT4 Floating Charts extension API.
 *
 * @license
 *  This content is released under the terms of the MIT license.
 *  A copy of this license has been included with this project in LICENSE.txt
 */
#property copyright "Copyright (c) 2015 TradertoolsFX"
#property link      "http://www.tradertools-fx.com"
#property strict

#include "fcerror.mqh"
#include "metatraderwindows.mqh"
#include "version.mqh"
#define _UNICODE // Used in windowmessages.mqh
#include "windowmessages.mqh"

/**
 * Class to interface with MT4 Floating Charts.
 */
class FloatingCharts {
public:
    FloatingCharts(void);
    int Init(void);

    /* Core float commands */
    int Float(long chart_id) const;
    int FloatAll(void) const;
    int Unfloat(long chart_id) const;
    int UnfloatAll(void) const;

    /* Custom toolbar actions */
    int CornerSnap(long chart_id, int corner) const;
    int PushBack(long chart_id) const;
    int ShowChartsSameSymbol(long chart_id) const;
    int StackChartsSameSymbol(long chart_id) const;

    /* Useful checks */
    bool IsFCRunning(void) const;
    int WaitForFC(uint timeout) const;
    bool IsChartFloating(long chart_id) const;

    /* Version info */
    string GetFCVersionString(void) const;
    string GetAPIVersionString(void) const;

    /* Version checks */
    bool RequiresFC(int major, int minor, int patch, bool do_alert) const;
private:
    const Version API_VERSION;

    int m_metatrader_window;
    int m_mdi_client_window;

    int SendToolbarCommand(long chart_id, int offset) const;
    int CheckCommand(int command_result) const;
    int SendCommand(int hwnd, int command) const;
    int GetFCVersion(void) const;
    int GetAPIVersion(void) const;
};

/**
 * Constructor.
 * Initializes constants and calls Init().
 */
FloatingCharts::FloatingCharts(void) : API_VERSION(1, 0, 0)
{
    Init();
}

/**
 * Initialize non-constant members.
 * @note
 * This method is called in the constructor for convenience.
 * However, it is safe to call it again to check for errors.
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns A non-zero #FCErrorCode if an error was encountered.
 */
int FloatingCharts::Init(void)
{
    int result = FC_ERR_SUCCESS;

    if (!IsDllsAllowed()) {
        result = FC_ERR_DLL_NOT_ALLOWED;
        m_mdi_client_window = 0;
        m_metatrader_window = 0;
    } else {
        m_mdi_client_window = GetMDIClientWindow();
        m_metatrader_window = GetMetaTraderWindow();

        if (m_mdi_client_window == 0 ||
                m_metatrader_window == 0) {
            result = FC_ERR_INVALID_HANDLE;
        }
    }

    return result;
}

/**
 * Float a single chart.
 *
 * @param long chart_id - chart id. 0 means current chart. @docked
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns A non-zero #FCErrorCode if an error was encountered.
 */
int FloatingCharts::Float(long chart_id) const
{
    int result = FC_ERR_SUCCESS;

    if (IsChartFloating(chart_id)) {
        result = FC_ERR_CHART_FLOATING;
    } else {
        int chart_frame_window = GetChartFrameWindow(chart_id);

        if (chart_frame_window == 0 ||
                m_mdi_client_window == 0 ||
                m_metatrader_window == 0) {
            result = FC_ERR_INVALID_HANDLE;
        } else {
            // The chart needs to be activated before floating it.
            SendMessage(m_mdi_client_window, WM_MDIACTIVATE,
                        chart_frame_window, 0);

            result = SendCommand(m_metatrader_window, COMMAND_ID_FLOAT);
        }
    }

    return result;
}

/**
 * Float all docked charts except for the currently active chart.
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns A non-zero #FCErrorCode if an error was encountered.
 */
int FloatingCharts::FloatAll(void) const
{
    return SendCommand(m_metatrader_window, COMMAND_ID_FLOAT_ALL);
}

/**
 * Unfloat a single chart.
 *
 * @param long chart_id - chart id. 0 means current chart. @undocked
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns A non-zero #FCErrorCode if an error was encountered.
 */
int FloatingCharts::Unfloat(long chart_id) const
{
    int result = FC_ERR_SUCCESS;

    if (!IsChartFloating(chart_id)) {
        result = FC_ERR_CHART_NOT_FLOATING;
    } else {
        int chart_frame_window = GetChartFrameWindow(chart_id);

        if (chart_frame_window == 0) {
            result = FC_ERR_INVALID_HANDLE;
        } else {
            SendMessage(chart_frame_window, WM_CLOSE, 0, 0);
        }
    }

    return result;
}

/**
 * Unfloat all floating charts.
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns A non-zero #FCErrorCode if an error was encountered.
 */
int FloatingCharts::UnfloatAll(void) const
{
    return SendCommand(m_metatrader_window, COMMAND_ID_UNFLOAT_ALL);
}

/**
 * Snap a floating chart to a screen corner.
 * Corresponds to the custom toolbar buttons 1,2,3,4.
 *
 * @param long chart_id - chart id. 0 means the current chart. @undocked
 * @param int corner - screen corner to snap to. Valid values are 0-3.
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns A non-zero #FCErrorCode if an error was encountered.
 *
 * @requiresfc 2.1.0
 */
int FloatingCharts::CornerSnap(long chart_id, int corner) const
{
    int result = FC_ERR_SUCCESS;

    if (corner > 3 || corner < 0) {
        result = FC_ERR_INVALID_PARAMETER;
    } else {
        result = SendToolbarCommand(chart_id, corner);
    }

    return result;
}

/**
 * Push a floating chart behind all windows.
 * Corresponds to the custom toolbar button 'PB'.
 *
 * @param long chart_id - chart id. 0 means the current chart. @undocked
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns A non-zero #FCErrorCode if an error was encountered.
 *
 * @requiresfc 2.4.0
 */
int FloatingCharts::PushBack(long chart_id) const
{
    return SendToolbarCommand(chart_id, PUSHBACK_OFFSET);
}

/**
 * Bring all floating charts with the same symbol as the given chart forward.
 * Corresponds to the custom toolbar button 'S*'.
 *
 * @param long chart_id - chart id. 0 means the current chart. @undocked
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns A non-zero #FCErrorCode if an error was encountered.
 *
 * @requiresfc 2.4.0
 */
int FloatingCharts::ShowChartsSameSymbol(long chart_id) const
{
    return SendToolbarCommand(chart_id, SYMBOL_MAPPING_OFFSET);
}

/**
 * Stack all floating charts with the same symbol as the given chart.
 * Corresponds to the custom toolbar button '[S]'.
 *
 * @param long chart_id - chart id. 0 means the current chart. @undocked
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns A non-zero #FCErrorCode if an error was encountered.
 *
 * @requiresfc 3.1.0
 */
int FloatingCharts::StackChartsSameSymbol(long chart_id) const
{
    return SendToolbarCommand(chart_id, SYMBOL_STACKING_OFFSET);
}

/**
 * Check if MT4 Floating Charts is running.
 *
 * @returns true if Floating Charts is attached to MetaTrader. Otherwise, false.
 *
 * @requiresfc 2.4.0
 */
bool FloatingCharts::IsFCRunning(void) const
{
    int info_res = SendMessage(m_metatrader_window,
                               WM_INFO_REQUEST,
                               WP_INFO_REQUEST_PING, 0);
    return (info_res == 1);
}

/**
 * Wait for MT4 Floating Charts to attach to MetaTrader.
 * Does not return until either MT4 Floating Charts
 * is detected or the timeout is reached.
 *
 * @param uint timeout - timeout in milliseconds. 0 means no timeout.
 *
 * @returns #FC_ERR_SUCCESS if MT4 Floating Charts is detected.
 * @returns #FC_ERR_NOT_RUNNING if the timeout is reached.
 *
 * @requiresfc 2.4.0
 */
int FloatingCharts::WaitForFC(unsigned int timeout) const
{
    int result = FC_ERR_SUCCESS;
    unsigned int start_ticks = GetTickCount();

    while (!IsFCRunning() &&
           !IsStopped()) {
        if (timeout > 0 &&
                timeout <= GetTickCount() - start_ticks) {
            result = FC_ERR_NOT_RUNNING;
            break;
        }
        Sleep(100);
    }

    return result;
}

/**
 * Check if a chart is floating.
 *
 * @param long chart_id - chart id. 0 means the current chart.
 *
 * @returns true if the given chart is floating. Otherwise, false.
 */
bool FloatingCharts::IsChartFloating(long chart_id) const
{
    int chart_frame_window = GetChartFrameWindow(chart_id);

    /* If the chart is floating,
     * the chart frame will not have a parent window. */
    return (GetParent(chart_frame_window) == 0);
}

/**
 * Get MT4 Floating Charts version as a string.
 * @note
 * If the version requirement is not met, then nothing is copied to str.
 *
 * @returns Version string as 'major.minor.patch'.
 * @returns NULL if Floating Charts version below 3.2.0.
 *
 * @requiresfc 3.2.0
 */
string FloatingCharts::GetFCVersionString(void) const
{
    string version_str = NULL;

    if (RequiresFC(3, 2, 0, false)) {
        Version fc_version(GetFCVersion());
        version_str = VersionToString(fc_version);
     }

     return version_str;
}

/**
 * Get the FloatingCharts API version as a string.
 *
 * @returns API version string as 'major.minor.patch'.
 */
string FloatingCharts::GetAPIVersionString(void) const
{
    return VersionToString(API_VERSION);
}

/**
 * Check if MT4 Floating Charts meets the minimum version requirement.
 * If the minimum version is less than 3.2.0,
 * the return value will always be true regardless of what version of FC
 * is actually running.
 *
 * @param int major - major part of version number (e.g., 3 in 3.4.5).
 * @param int minor - minor part of version number (e.g., 4 in 3.4.5).
 * @param int patch - patch part of version (e.g., 5 in 3.4.5).
 * @param bool do_alert - if true, a standard alert is shown if
 *                        the minimum version requirement is not met.
 *
 * @returns true if MT4 Floating Charts is >= minimum version. Otherwise, false.
 *
 * @requiresfc 3.2.0
 */
bool FloatingCharts::RequiresFC(int major, int minor, int patch,
                                bool do_alert) const
{
    Version minimum_test_version(3, 2, 0);
    Version minimum_version(major, minor, patch);
    Version fc_version(GetFCVersion());

    if (VersionCompare(minimum_version, minimum_test_version) < 0) {
        // Cannot test for version prior to 3.2.0
        return true;
    } else if (VersionCompare(fc_version, minimum_version) >= 0) {
        return true;
    } else if (do_alert) {
        Alert(
            StringFormat(
                "This script requires MT4 Floating Charts version %s or higher!",
                VersionToString(minimum_version)
            )
        );
    }

    return false;
}

/**
 * @internal
 * Send custom toolbar command to a floating chart.
 *
 * @param long chart_id - chart id. 0 means the current chart. @undocked
 * @param int offset - custom toolbar button offset relative to FIRST_BUTTON_ID.
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns A non-zero #FCErrorCode if an error was encountered.
 */
int FloatingCharts::SendToolbarCommand(long chart_id, int offset) const
{
    int result = FC_ERR_SUCCESS;

    if (!IsChartFloating(chart_id)) {
        result = FC_ERR_CHART_NOT_FLOATING;
    } else {
        int chart_frame_window = GetChartFrameWindow(chart_id);
        int command_id = COMMAND_ID_FIRST_BUTTON + offset;
        result = SendCommand(chart_frame_window, command_id);
    }

    return result;
}

/**
 * @internal
 * Check the result of an MT4 Floating Charts command.
 * If the MT4 Floating Charts version is less than 3.2.0,
 * it will return FC_ERR_SUCCESS no matter what.
 *
 * @param int command_result
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns #FC_ERR_COMMAND_FAILED on failure.
 *
 * @requiresfc 3.2.0
 */
int FloatingCharts::CheckCommand(int command_result) const
{
    int result = FC_ERR_SUCCESS;

    Version fc_version(GetFCVersion());
    Version minimum_version(3, 2, 0);

    if (VersionCompare(fc_version, minimum_version) >= 0 &&
            command_result == 0) {
        result = FC_ERR_COMMAND_FAILED;
    }

    return result;
}

/**
 * @internal
 * Send an MT4 Floating Charts command.
 * If the MT4 Floating Charts version is less than 3.2.0,
 * it will return FC_ERR_SUCCESS no matter what.
 *
 * @param int hwnd - window to send the command to.
 * @param int command
 *
 * @returns #FC_ERR_SUCCESS on success.
 * @returns A non-zero #FCErrorCode on failure.
 */
int FloatingCharts::SendCommand(int hwnd, int command) const
{
    int result = FC_ERR_SUCCESS;

    if (hwnd == 0) {
        result = FC_ERR_INVALID_HANDLE;
    } else {
        int cmd_res = SendMessage(hwnd, WM_COMMAND, command, 0);
        result = CheckCommand(cmd_res);
    }

    return result;
}

/**
 * @internal
 * Get MT4 Floating Charts version in the form of a calculated integer.
 * @note
 * Do not bother using this unless you are trying to make sure
 * the user is using a version greater than or equal to 3.2.0.
 * MT4 Floating Charts will return 0 for its version on any
 * version below 3.2.0.
 *
 * @returns MT4 Floating Charts version integer.
 *
 * @requiresfc 3.2.0
 */
int FloatingCharts::GetFCVersion(void) const
{
    return SendMessage(m_metatrader_window, WM_INFO_REQUEST,
                       WP_INFO_REQUEST_VERSION, 0);
}

/**
 * @internal
 * Get the FloatingCharts API version as a calculated integer.
 *
 * @returns API version integer.
 */
int FloatingCharts::GetAPIVersion(void) const
{
    return VersionToInt(API_VERSION);
}
