/* FC_SaveFloatProfile.mq4 - Save default or named float profile.
 * Regardless of whether you save the "default" profile or a named profile,
 * the float profile is tied to the MetaTrader profile.
 * For example, if you have a MetaTrader profile named "Market Overview"
 * and you save a float profile named "EURUSD", the float profile is
 * effectively "Market Overview_EURUSD". If you switch to another profile
 * called "Weekly" and try to load the float profile "EURUSD", it will not load
 * the profile from "Market Overview", rather, fail or load "Weekly_EURUSD" if
 * you have ever saved a float profile as "EURUSD" on the "Weekly" profile.
 * Note that it is not actually tied to the MT4 profile name, rather, an id,
 * so you can change names without a problem.
 *
 * This content is released under the terms of the MIT license.
 * A copy of this license has been included with this project in LICENSE.txt
 */
#property copyright   "Copyright (c) 2015 TradertoolsFX"
#property link        "http://www.tradertools-fx.com"
#property strict

//#define LASTSTATE // If defined, hide inputs and and just save default profile.
#ifndef LASTSTATE
#property show_inputs
#endif

#include "../inc/fcprecheck.mqh"
#include "../inc/fcprofiles.mqh"

input string _desc_ = "== Leave ProfileName blank to save as the default float profile ==";
input string ProfileName = "";

void OnStart()
{
    FloatingCharts fc;
    if (FCPreCheck(fc) != FC_ERR_SUCCESS ||
            !fc.RequiresFC(3, 2, 0, true)) {
        return;
    }

    // Check if there is anything to save
    int count = GetFloatingChartCount(fc);
    if (count == 0) {
        Alert("No charts are floating. Nothing to save ...");
        return;
    }

    // Open file for writing
    ResetLastError();
    int file_handle = FileOpen(GetFileName(GetProfileId(fc), ProfileName),
                               FILE_WRITE|FILE_BIN);
    if (file_handle == INVALID_HANDLE) {
        Alert("Failed to open file for saving! Last error: ", GetLastError());
        return;
    }

    long chart_ids[];
    if (ArrayResize(chart_ids, count) == count) {
        StoreChartIds(chart_ids, count, fc);

        // Write chart ids array to file
        ResetLastError();
        if (FileWriteArray(file_handle, chart_ids) == 0) {
            Print("Failed to write to file. Last Error: ", GetLastError());
        }

        // Cleanup
        FileFlush(file_handle);
        FileClose(file_handle);

        // Report status to user
        Alert("Float profile saved successfully!");
    } else {
        Alert("Failed to resize array chart_ids. Last error: ",
              GetLastError());
    }
}
