/**
 * fctiling.mqh
 *
 * This content is released under the terms of the MIT license.
 * A copy of this license has been included with this project in LICENSE.txt
 */
#property copyright   "Copyright (c) 2015 TradertoolsFX"
#property link        "http://www.tradertools-fx.com"
#property strict

#include "../../api/FloatingCharts.mqh"
#include "MapArrayInt.mqh"

#define DWORD    unsigned int
#define LONG     int

/* Windows' RECT structure. */
struct RECT {
    LONG left;
    LONG top;
    LONG right;
    LONG bottom;
};

/* Windows' MONITORINFO structure. */
struct MONITORINFO {
    DWORD cbSize;
    RECT rcMonitor;
    RECT rcWork;
    DWORD dwFlags;
};

#define MONITOR_DEFAULTTONEAREST 0x00000002
#define HWND_TOP 0
#define SWP_NOZORDER 0x0004
#define SW_RESTORE 9

#import "user32.dll"
    int GetWindowRect(int hWnd, RECT &lpRect);
    int MonitorFromWindow(int hwnd, DWORD dwFlags);
    int GetMonitorInfoW(int hMonitor, MONITORINFO &lpmi);
#import

struct TileLayout {
    int columns;
    int rows;
};

/**
 * Get the tile layout given the count of charts on one monitor.
 * Layout hard coded for up to 16 charts:
 *      columnXrow
 *  1 - 1x1
 *  2 - 2x1
 *  3 - 3x1
 *  4 - 2x2
 *  6 - 3x2
 *  8 - 4x2
 *  9 - 3x3
 * 12 - 4x3
 * 16 - 4x4
 * 17+ - calculated
 *
 * @param[in] TileLayout layout
 * @param int chart_count
 */
void GetTileLayout(TileLayout &layout, int chart_count)
{
    if (chart_count < 4) { // 1-3
        layout.columns = chart_count;
        layout.rows = 1;
    } else if (chart_count == 4) { // 4
        layout.columns = 2;
        layout.rows = 2;
    } else if (chart_count <= 6) { // 5-6
        layout.columns = 3;
        layout.rows = 2;
    } else if (chart_count <= 8) { // 7-8
        layout.columns = 4;
        layout.rows = 2;
    } else if (chart_count == 9) { // 9
        layout.columns = 3;
        layout.rows = 3;
    } else if (chart_count <= 12) { // 10-12
        layout.columns = 4;
        layout.rows = 3;
    } else if (chart_count <= 16) { // 13-16
        layout.columns = 4;
        layout.rows = 4;
    } else { // 17+
        layout.rows = MathMax(MathLog2(chart_count), 1);
        layout.columns = MathMax(MathCeil(chart_count / layout.rows), 1);

        int product = layout.columns * layout.rows;
        if (product < chart_count) {
           layout.columns += MathSqrt(chart_count - product);
        }
    }
}

void TileChart(int chart, const MONITORINFO &mi,
               const TileLayout &layout,
               int column,
               int row)
{
    LONG x = mi.rcWork.left;
    LONG y = mi.rcWork.top;
    LONG w = (mi.rcWork.right - x) / layout.columns;
    LONG h = (mi.rcWork.bottom - y) / layout.rows;
    x += (w * column);
    y += (h * row);

    ShowWindow(chart, SW_RESTORE);
    SetWindowPos(chart, HWND_TOP, x, y, w, h, SWP_NOZORDER);
}

void TileFloatingCharts(const MapArrayInt &monitor_chart_map)
{
    int num_monitors = monitor_chart_map.GetKeyCount();
    int chart, monitor;
    int num_charts;
    int row, column;
    TileLayout layout;
    MONITORINFO mi;

    for (int i = 0; i < num_monitors; i++) {
        row = column = 0;

        ZeroMemory(mi);
        mi.cbSize = sizeof(mi);

        monitor = monitor_chart_map.GetKey(i);
        num_charts = monitor_chart_map.GetValueCount(monitor);

        if (GetMonitorInfoW(monitor, mi)) {
            GetTileLayout(layout, num_charts);
            for (int j = 0; j < num_charts; j++) {
                chart = monitor_chart_map.GetValue(monitor, j);
                TileChart(chart, mi, layout, column, row);

                column++;
                if (column == layout.columns) {
                    column = 0;
                    row++;
                }
            }
        }
    }
}

double MathLog2(double value)
{
    return MathLog(value) / MathLog(2);
}
