/* fcprofiles.mqh - Functions shared by save/load profile scripts.
 *
 * This content is released under the terms of the MIT license.
 * A copy of this license has been included with this project in LICENSE.txt
 */
#property copyright   "Copyright (c) 2015 TradertoolsFX"
#property link        "http://www.tradertools-fx.com"
#property strict

#import "user32.dll"
    int RegisterWindowMessageW(string lpString);
#import

#include "../../api/FloatingCharts.mqh"

#define WP_INFO_REQUEST_PROFILE_ID 550

/**
 * Get the current MetaTrader profile id from MT4 Floating Charts.
 *
 * @param FloatingCharts fc
 *
 * @returns The profile id (positive integer).
 * @returns -1 on failure.
 *
 * @requiresfc 3.2.0
 */
int GetProfileId(const FloatingCharts &fc)
{
    if (!fc.RequiresFC(3, 2, 0, true)) {
        return -1;
    }

    int metatrader_window = GetMetaTraderWindow();
    if (metatrader_window == 0) {
        Print(__FUNCTION__, ": Unable to get metatrader window handle!");
        return -1;
    }

    return SendMessage(metatrader_window, WM_INFO_REQUEST,
                       WP_INFO_REQUEST_PROFILE_ID, 0);
}

/**
 * @param int profile_id - The MetaTrader profile.
 * @param string profile - The name of the float profile being saved.
 *                         Leave blank for the general "last state" profile.
 */
string GetFileName(int profile_id, string profile="")
{
    const string prefix = "fc_float_state_";
    return StringConcatenate(prefix, profile_id, profile);
}

int GetFloatingChartCount(const FloatingCharts &fc)
{
    int count = 0;
    long chart_id = ChartFirst();
    while (chart_id != -1 &&
           !IsStopped()) {
        if (fc.IsChartFloating(chart_id)) {
            count++;
        }

        chart_id = ChartNext(chart_id);
    }

    return count;
}

/**
 * Find all floating charts and save their ids in an array.
 *
 * @param[out] long chart_ids[]
 * @param int max - the max number of ids to copy to chart_ids.
 * @param FloatingCharts fc
 */
void StoreChartIds(long &chart_ids[], int max, const FloatingCharts &fc)
{
    long chart_id = ChartFirst();
    int i = 0;
    while (chart_id != -1 &&
           i < max &&
           !IsStopped()) {
        if (fc.IsChartFloating(chart_id)) {
            chart_ids[i++] = chart_id;
        }

        chart_id = ChartNext(chart_id);
    }
}
