/**
 * @file fcprecheck.mqh
 * My custom FloatingCharts API precheck functions.
 *
 * @license
 *  This content is released under the terms of the MIT license.
 *  A copy of this license has been included with this project in LICENSE.txt
 */
#property copyright   "Copyright (c) 2015 TradertoolsFX"
#property link        "http://www.tradertools-fx.com"
#property strict

#include "../../api/FloatingCharts.mqh"

/* NOTE: It is useless to check MT4 build because these scripts
 * will only compile on build 600+.
 * Therefore, they will not even run on earlier builds. */

/**
 * Perform FloatingCharts API prechecks and alert user on failure.
 *
 * @param FloatingCharts fc
 *
 * @returns FC_ERR_SUCCESS (0) on success.
 * @returns A non-zero FCErrorCode if an error was encountered.
 */
int FCPreCheck(const FloatingCharts &fc)
{
    int result = FC_ERR_SUCCESS;

    if (!IsDllsAllowed()) {
        Alert("DLLs are not allowed!\n",
              "This script cannot communicate with MT4 Floating Charts unless\n",
              "Tools > Expert Advisors > 'Allow DLL Imports' is checked.");
        result = FC_ERR_DLL_NOT_ALLOWED;
    } else if (!fc.IsFCRunning()) {
        Alert("MT4 Floating Charts is not running!\n",
              "This script only works with MT4 Floating Charts running.");
        result = FC_ERR_NOT_RUNNING;
    }

    return result;
}

/**
 * Perform FloatingCharts API prechecks and wait for MT4 Floating Charts.
 * Waits up to 3 minutes for MT4 Floating Charts to attach
 * before returning FC_ERR_NOT_RUNNING.
 *
 * @param FloatingCharts fc
 *
 * @returns FC_ERR_SUCCESS (0) on success.
 * @returns A non-zero FCErrorCode if an error was encountered.
 */
int FCPreCheckWait(const FloatingCharts &fc)
{
    int result = FC_ERR_SUCCESS;
    /* MT4 Floating Charts does not initially attach to MetaTrader
     * until the user clicks on the MetaTrader window OR
     * a certain amount of time has passed. Depending on the exact
     * timing of things, MT4 Floating Charts may wait up to 3 minutes
     * before attaching if the user never clicks on MetaTrader. */
    int timeout = 1000 * 60 * 3; // 3 minutes

    if (!IsDllsAllowed()) {
        Alert("DLLs are not allowed!\n",
              "This script cannot communicate with MT4 Floating Charts unless\n",
              "Tools > Expert Advisors > 'Allow DLL Imports' is checked.");
        result = FC_ERR_DLL_NOT_ALLOWED;
    }

    if (fc.WaitForFC(timeout) != FC_ERR_SUCCESS) {
        Alert("MT4 Floating Charts is not running!\n",
              "This script only works with MT4 Floating Charts running.");
        result = FC_ERR_NOT_RUNNING;
    }

    return result;
}
