/* FC_TileFloatingCharts.mq4 - Tile floating charts across monitor(s).
 *
 * This content is released under the terms of the MIT license.
 * A copy of this license has been included with this project in LICENSE.txt
 */
#property copyright   "Copyright (c) 2015 TradertoolsFX"
#property link        "http://www.tradertools-fx.com"
#property version     "1.0"
#property strict
#property show_inputs

#include "inc/fcprecheck.mqh"
#include "inc/fctiling.mqh"

input int MaxChartsPerMonitor = 8;

void OnStart()
{
    FloatingCharts fc;
    if (FCPreCheck(fc) != FC_ERR_SUCCESS) {
        return;
    }

    CArrayInt chart_frame_windows;
    GetFloatingChartWindows(chart_frame_windows, fc);

    MapArrayInt monitor_chart_map;
    MapMonitorsToFloatingCharts(monitor_chart_map, chart_frame_windows);

    TileFloatingCharts(monitor_chart_map);
}

/**
 * Get floating chart frame window handles.
 *
 * @param[out] CArrayInt chart_frame_windows
 * @param[in] FloatingCharts fc
 *
 * @returns Count of floating charts.
 */
int GetFloatingChartWindows(CArrayInt &chart_frame_windows,
                            const FloatingCharts &fc)
{
    int count = 0;
    long chart_id = ChartFirst();

    while (chart_id != -1 &&
           !IsStopped()) {
        if (fc.IsChartFloating(chart_id)) {
            chart_frame_windows.Add(GetChartFrameWindow(chart_id));
            count++;
        }

        chart_id = ChartNext(chart_id);
    }

    chart_frame_windows.Sort();

    return count;
}

/**
 * Get monitor handles.
 *
 * @param[out] int handles[]
 *
 * @returns Count of monitors.
 */
int GetMonitors(int &handles[])
{
    int monitor;
    string class_name = NULL;
    string window_name = NULL;

    int num_monitors = GetSystemMetrics(SM_CMONITORS);
    if (ArrayResize(handles, num_monitors) == -1) {
        Print(__FUNCTION__,
              ": Failed to resize array. Last error: ",
              GetLastError());
        return -1;
    }
    ArrayInitialize(handles, 0);

    int count = 0;
    bool found = false;
    int next_window = FindWindowW(class_name, window_name);

    while (next_window != 0 &&
           count < num_monitors &&
           !IsStopped()) {
        found = false;
        monitor = MonitorFromWindow(next_window, MONITOR_DEFAULTTONEAREST);

        for (int i = 0; i < count; i++) {
            if (monitor == handles[i]) {
                found = true;
                break;
            }
        }

        if (!found) {
            handles[count++] = monitor;
        }

        next_window = FindWindowExW(0, next_window, class_name, window_name);
    }

    ArraySort(handles);

    return count;
}

/**
 * Create a monitor => charts[] map.
 * Each monitor handle is a key and maps to an array of chart frame windows.
 *
 * @param[out] MapArrayInt map
 * @param[in] CArrayInt chart_frame_windows
 */
void MapMonitorsToFloatingCharts(MapArrayInt &map,
                                 const CArrayInt &chart_frame_windows)
{
    int monitors[];
    int num_monitors = GetMonitors(monitors);
    int num_charts = chart_frame_windows.Total();
    int im = 0;
    int chart_count = 0;

    for (int i = 0; i < num_charts; i++) {
        map.AddValue(monitors[im], chart_frame_windows[i]);
        chart_count++;

        if (chart_count == MaxChartsPerMonitor) {
            im++;
            chart_count = 0;
            if (im >= num_monitors) {
                break;
            }
        }
    }
}
