/**
 * @internal
 * @file metatraderwindows.mqh
 * Convenience functions to get important MetaTrader window handles.
 *
 * @license
 *  This content is released under the terms of the MIT license.
 *  A copy of this license has been included with this project in LICENSE.txt
 */
#property copyright "Copyright (c) 2015 TradertoolsFX"
#property link      "http://www.tradertools-fx.com"
#property strict

#include <WinUser32.mqh>

#import "user32.dll"
    int FindWindowExW(int hwndParent,
                      int hwndChildAfter,
                      string &lpszClass,
                      string &lpszWindow);
    uint GetWindowThreadProcessId(int hWnd, uint &lpdwProcessId);
#import

#import "kernel32.dll"
    uint GetCurrentProcessId(void);
#import

#define METATRADER4_CLASS  "MetaQuotes::MetaTrader::4.00"
#define MDI_CLIENT_CLASS   "MDIClient"

/**
 * Get a chart's window handle.
 *
 * @param long chart_id - chart id. 0 means the current chart.
 *
 * @returns The chart window handle.
 */
int GetChartWindow(long chart_id)
{
    // Explicit cast to suppress compiler warnings
    return (int)ChartGetInteger(chart_id, CHART_WINDOW_HANDLE);
}

/**
 * Get a chart's frame window.
 *
 * @param long chart_id - chart id. 0 means the current chart.
 *
 * @returns The chart frame window handle on success.
 * @returns 0 on failure.
 */
int GetChartFrameWindow(long chart_id)
{
    return GetParent(GetChartWindow(chart_id));
}

/**
 * Get the MDI client window handle.
 *
 * @returns The MDI client window handle on success.
 * @returns 0 on failure.
 */
int GetMDIClientWindow(void)
{
    int hwnd = GetMetaTraderWindow();
    string window_class = MDI_CLIENT_CLASS;
    string window_name = NULL;

    return FindWindowExW(hwnd, 0, window_class, window_name);
}

/**
 * Get MetaTrader's window handle.
 *
 * @returns The MetaTrader window handle on success.
 * @returns 0 on failure.
 */
int GetMetaTraderWindow(void)
{
    /* NOTE:
     * GetAncestor(chart_window, GA_ROOT) does not return the correct handle
     * if the chart is floating, so a different method has to be used.
     */
    string window_class = METATRADER4_CLASS;
    string window_name = NULL;

    unsigned int process_id = GetCurrentProcessId();
    unsigned int metatrader_process_id = 0;

    /* Loop through MetaTrader windows
     * until one is found with the same process id. */
    int hwnd = FindWindowW(window_class, window_name);
    while (hwnd != 0) {
        metatrader_process_id = 0;
        GetWindowThreadProcessId(hwnd, metatrader_process_id);
        if (metatrader_process_id == process_id) {
            // Found the right window.
            return hwnd;
        }

        hwnd = FindWindowExW(0, hwnd, window_class, window_name);
    }

    // Failure
    return 0;
}
